"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.exiftool = exports.ExifTool = exports.UnsetZoneOffsetMinutes = exports.UnsetZoneName = exports.UnsetZone = exports.offsetMinutesToZoneName = exports.defaultVideosToUTC = exports.parseJSON = exports.isIgnorableWarning = exports.ExifToolTask = exports.ExifTime = exports.ExifDateTime = exports.ExifDate = exports.BinaryField = void 0;
const bc = __importStar(require("batch-cluster"));
const _cp = __importStar(require("child_process"));
const _fs = __importStar(require("fs"));
const process_1 = __importDefault(require("process"));
const AsyncRetry_1 = require("./AsyncRetry");
const BinaryExtractionTask_1 = require("./BinaryExtractionTask");
const BinaryToBufferTask_1 = require("./BinaryToBufferTask");
const DefaultExifToolOptions_1 = require("./DefaultExifToolOptions");
const DeleteAllTagsArgs_1 = require("./DeleteAllTagsArgs");
const IsWin32_1 = require("./IsWin32");
const Lazy_1 = require("./Lazy");
const Pick_1 = require("./Pick");
const ReadRawTask_1 = require("./ReadRawTask");
const ReadTask_1 = require("./ReadTask");
const RewriteAllTagsTask_1 = require("./RewriteAllTagsTask");
const String_1 = require("./String");
const VersionTask_1 = require("./VersionTask");
const WriteTask_1 = require("./WriteTask");
var BinaryField_1 = require("./BinaryField");
Object.defineProperty(exports, "BinaryField", { enumerable: true, get: function () { return BinaryField_1.BinaryField; } });
var ExifDate_1 = require("./ExifDate");
Object.defineProperty(exports, "ExifDate", { enumerable: true, get: function () { return ExifDate_1.ExifDate; } });
var ExifDateTime_1 = require("./ExifDateTime");
Object.defineProperty(exports, "ExifDateTime", { enumerable: true, get: function () { return ExifDateTime_1.ExifDateTime; } });
var ExifTime_1 = require("./ExifTime");
Object.defineProperty(exports, "ExifTime", { enumerable: true, get: function () { return ExifTime_1.ExifTime; } });
var ExifToolTask_1 = require("./ExifToolTask");
Object.defineProperty(exports, "ExifToolTask", { enumerable: true, get: function () { return ExifToolTask_1.ExifToolTask; } });
var IgnorableError_1 = require("./IgnorableError");
Object.defineProperty(exports, "isIgnorableWarning", { enumerable: true, get: function () { return IgnorableError_1.isIgnorableWarning; } });
var JSON_1 = require("./JSON");
Object.defineProperty(exports, "parseJSON", { enumerable: true, get: function () { return JSON_1.parseJSON; } });
var Timezones_1 = require("./Timezones");
Object.defineProperty(exports, "defaultVideosToUTC", { enumerable: true, get: function () { return Timezones_1.defaultVideosToUTC; } });
Object.defineProperty(exports, "offsetMinutesToZoneName", { enumerable: true, get: function () { return Timezones_1.offsetMinutesToZoneName; } });
Object.defineProperty(exports, "UnsetZone", { enumerable: true, get: function () { return Timezones_1.UnsetZone; } });
Object.defineProperty(exports, "UnsetZoneName", { enumerable: true, get: function () { return Timezones_1.UnsetZoneName; } });
Object.defineProperty(exports, "UnsetZoneOffsetMinutes", { enumerable: true, get: function () { return Timezones_1.UnsetZoneOffsetMinutes; } });
const _ignoreShebang = (0, Lazy_1.lazy)(() => !(0, IsWin32_1.isWin32)() && !_fs.existsSync("/usr/bin/perl"));
/**
 * Manages delegating calls to a vendored running instance of ExifTool.
 *
 * Instances should be shared: consider using the exported singleton
 * instance of this class, `exiftool`.
 */
class ExifTool {
    constructor(options = {}) {
        var _a;
        /**
         * Register life cycle event listeners. Delegates to BatchProcess.
         */
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        this.on = (event, listener) => this.batchCluster.on(event, listener);
        if (options != null && typeof options !== "object") {
            throw new Error("Please update caller to the new ExifTool constructor API");
        }
        const o = {
            ...DefaultExifToolOptions_1.DefaultExifToolOptions,
            ...options,
        };
        const ignoreShebang = (_a = o.ignoreShebang) !== null && _a !== void 0 ? _a : _ignoreShebang();
        const env = { ...o.exiftoolEnv, LANG: "C" };
        if ((0, String_1.notBlank)(process_1.default.env.EXIFTOOL_HOME) && (0, String_1.blank)(env.EXIFTOOL_HOME)) {
            env.EXIFTOOL_HOME = process_1.default.env.EXIFTOOL_HOME;
        }
        const spawnOpts = {
            stdio: "pipe",
            shell: ignoreShebang,
            detached: false,
            env,
        };
        const processFactory = () => ignoreShebang
            ? _cp.spawn("perl", [o.exiftoolPath, ...o.exiftoolArgs], spawnOpts)
            : _cp.spawn(o.exiftoolPath, o.exiftoolArgs, spawnOpts);
        this.options = {
            ...o,
            ignoreShebang,
            processFactory,
            exitCommand: o.exitCommand,
            versionCommand: o.versionCommand,
            // User options win:
            ...options,
        };
        this.batchCluster = new bc.BatchCluster(this.options);
    }
    /**
     * @return a promise holding the version number of the vendored ExifTool
     */
    version() {
        return this.enqueueTask(() => new VersionTask_1.VersionTask());
    }
    /**
     * Read the tags in `file`.
     *
     * @param {string} file the file to extract metadata tags from
     *
     * @param {string[]} [optionalArgs] any additional ExifTool arguments, like
     * "-fast" or "-fast2". **Most other arguments will require you to use
     * `readRaw`.** Note that the default is "-fast", so if you want ExifTool to
     * read the entire file for metadata, you should pass an empty array as the
     * second parameter. See https://exiftool.org/#performance for more
     * information about `-fast` and `-fast2`.
     *
     * @returns {Promise<Tags>} A resolved Tags promise. If there are errors
     * during reading, the `.errors` field will be present.
     */
    read(file, optionalArgs = ["-fast"], options) {
        return this.enqueueTask(() => ReadTask_1.ReadTask.for(file, {
            optionalArgs,
            ...(0, Pick_1.pick)(this.options, "numericTags", "useMWG", "includeImageDataMD5", "defaultVideosToUTC", "geoTz"),
            ...options,
        })); // < no way to know at compile time if we're going to get back a T!
    }
    /**
     * Read the tags from `file`, without any post-processing of ExifTool
     * values.
     *
     * **You probably want `read`, not this method. READ THE REST OF THIS
     * COMMENT CAREFULLY.**
     *
     * If you want to extract specific tag values from a file, you may want to
     * use this, but all data validation and inference heuristics provided by
     * `read` will be skipped.
     *
     * Note that performance will be very similar to `read`, and will actually
     * be worse if you don't include `-fast` or `-fast2` (as the most expensive
     * bit is the perl interpreter and scanning the file on disk).
     *
     * @param args any additional arguments other than the file path. Note that
     * "-json", and the Windows unicode filename handler flags, "-charset
     * filename=utf8", will be added automatically.
     *
     * @return Note that the return value will be similar to `Tags`, but with no
     * date, time, or other rich type parsing that you get from `.read()`. The
     * field values will be `string | number | string[]`.
     *
     * @see https://github.com/photostructure/exiftool-vendored.js/issues/44 for
     * typing details.
     */
    readRaw(file, args = []) {
        return this.enqueueTask(() => ReadRawTask_1.ReadRawTask.for(file, args));
    }
    /**
     * Write the given `tags` to `file`.
     *
     * @param {string} file an existing file to write `tags` to.
     * @param {Tags} tags the tags to write to `file`.
     * @param {string[]} [args] any additional ExifTool arguments, like "-n".
     * Most consumers won't probably need this.
     * @returns {Promise<void>} Either the promise will be resolved if the
     * tags are written to successfully, or the promise will be rejected if
     * there are errors or warnings.
     * @memberof ExifTool
     */
    write(file, tags, args, options) {
        const retriable = false;
        return this.enqueueTask(() => WriteTask_1.WriteTask.for(file, tags, args, {
            ...(0, Pick_1.pick)(this.options, "useMWG"),
            ...options,
        }), retriable);
    }
    /**
     * This will strip `file` of all metadata tags. The original file (with the
     * name `${FILENAME}_original`) will be retained. Note that some tags, like
     * stat information and image dimensions, are intrinsic to the file and will
     * continue to exist if you re-`read` the file.
     */
    deleteAllTags(file) {
        return this.write(file, {}, DeleteAllTagsArgs_1.DeleteAllTagsArgs);
    }
    /**
     * Extract the low-resolution thumbnail in `path/to/image.jpg`
     * and write it to `path/to/thumbnail.jpg`.
     *
     * Note that these images can be less than .1 megapixels in size.
     *
     * @return a `Promise<void>`. An `Error` is raised if
     * the file could not be read or the output not written.
     */
    extractThumbnail(imageFile, thumbnailFile) {
        return this.extractBinaryTag("ThumbnailImage", imageFile, thumbnailFile);
    }
    /**
     * Extract the "preview" image in `path/to/image.jpg`
     * and write it to `path/to/preview.jpg`.
     *
     * The size of these images varies widely, and is present in dSLR images.
     * Canon, Fuji, Olympus, and Sony use this tag.
     *
     * @return a `Promise<void>`. An `Error` is raised if
     * the file could not be read or the output not written.
     */
    extractPreview(imageFile, previewFile) {
        return this.extractBinaryTag("PreviewImage", imageFile, previewFile);
    }
    /**
     * Extract the "JpgFromRaw" image in `path/to/image.jpg` and write it to
     * `path/to/fromRaw.jpg`.
     *
     * This size of these images varies widely, and is not present in all RAW
     * images. Nikon and Panasonic use this tag.
     *
     * @return a `Promise<void>`. The promise will be rejected if the file could
     * not be read or the output not written.
     */
    extractJpgFromRaw(imageFile, outputFile) {
        return this.extractBinaryTag("JpgFromRaw", imageFile, outputFile);
    }
    /**
     * Extract a given binary value from "tagname" tag associated to
     * `path/to/image.jpg` and write it to `dest` (which cannot exist and whose
     * directory must already exist).
     *
     * @return a `Promise<void>`. The promise will be rejected if the binary
     * output not be written to `dest`.
     */
    async extractBinaryTag(tagname, src, dest) {
        // BinaryExtractionTask returns a stringified error if the output indicates
        // the task should not be retried.
        const maybeError = await this.enqueueTask(() => BinaryExtractionTask_1.BinaryExtractionTask.for(tagname, src, dest));
        if (maybeError != null) {
            throw new Error(maybeError);
        }
    }
    /**
     * Extract a given binary value from "tagname" tag associated to
     * `path/to/image.jpg` as a `Buffer`. This has the advantage of not writing to
     * a file, but if the payload associated to `tagname` is large, this can cause
     * out-of-memory errors.
     *
     * @return a `Promise<Buffer>`. The promise will be rejected if the file or
     * tag is missing.
     */
    async extractBinaryTagToBuffer(tagname, imageFile) {
        const result = await this.enqueueTask(() => BinaryToBufferTask_1.BinaryToBufferTask.for(tagname, imageFile));
        if (Buffer.isBuffer(result)) {
            return result;
        }
        else if (result instanceof Error) {
            throw result;
        }
        else {
            throw new Error("Unexpected result from BinaryToBufferTask: " + JSON.stringify(result));
        }
    }
    /**
     * Attempt to fix metadata problems in JPEG images by deleting all metadata
     * and rebuilding from scratch. After repairing an image you should be able to
     * write to it without errors, but some metadata from the original image may
     * be lost in the process.
     *
     * This should only be applied as a last resort to images whose metadata is
     * not readable via {@link .read()}.
     *
     * @see https://exiftool.org/faq.html#Q20
     *
     * @param {string} inputFile the path to the problematic image
     * @param {string} outputFile the path to write the repaired image
     * @param {boolean} allowMakerNoteRepair if there are problems with MakerNote
     * tags, allow ExifTool to apply heuristics to recover corrupt tags. See
     * exiftool's `-F` flag.
     * @return {Promise<void>} resolved when outputFile has been written.
     */
    rewriteAllTags(inputFile, outputFile, allowMakerNoteRepair = false) {
        return this.enqueueTask(() => RewriteAllTagsTask_1.RewriteAllTagsTask.for(inputFile, outputFile, allowMakerNoteRepair));
    }
    /**
     * Shut down running ExifTool child processes. No subsequent requests will be
     * accepted.
     *
     * This may need to be called in `after` or `finally` clauses in tests or
     * scripts for them to exit cleanly.
     */
    end(gracefully = true) {
        return this.batchCluster.end(gracefully).promise;
    }
    /**
     * @return true if `.end()` has been invoked
     */
    get ended() {
        return this.batchCluster.ended;
    }
    /**
     * Most users will not need to use `enqueueTask` directly. This method
     * supports submitting custom `BatchCluster` tasks.
     *
     * @see BinaryExtractionTask for an example task implementation
     */
    enqueueTask(task, retriable = true) {
        const f = async () => {
            const t = await task();
            // if we have to add more options for every task, rethink this approach:
            t.isIgnorableError = this.options.isIgnorableError;
            return this.batchCluster.enqueueTask(t);
        };
        return retriable ? (0, AsyncRetry_1.retryOnReject)(f, this.options.taskRetries) : f();
    }
    /**
     * @return the currently running ExifTool processes. Note that on Windows,
     * these are only the process IDs of the directly-spawned ExifTool wrapper,
     * and not the actual perl vm. This should only really be relevant for
     * integration tests that verify processes are cleaned up properly.
     */
    get pids() {
        return this.batchCluster.pids();
    }
    /**
     * @return the number of pending (not currently worked on) tasks
     */
    get pendingTasks() {
        return this.batchCluster.pendingTaskCount;
    }
    /**
     * @return the total number of child processes created by this instance
     */
    get spawnedProcs() {
        return this.batchCluster.spawnedProcCount;
    }
    /**
     * @return the current number of child processes currently servicing tasks
     */
    get busyProcs() {
        return this.batchCluster.busyProcCount;
    }
    /**
     * @return report why child processes were recycled
     */
    childEndCounts() {
        return this.batchCluster.childEndCounts;
    }
    /**
     * Shut down any currently-running child processes. New child processes will
     * be started automatically to handle new tasks.
     */
    closeChildProcesses(gracefully = true) {
        return this.batchCluster.closeChildProcesses(gracefully);
    }
}
exports.ExifTool = ExifTool;
/**
 * Use this singleton rather than instantiating new ExifTool instances in order
 * to leverage a single running ExifTool process. As of v3.0, its `maxProcs` is
 * set to the number of CPUs on the current system; no more than `maxProcs`
 * instances of `exiftool` will be spawned. You may want to experiment with
 * smaller or larger values for `maxProcs`, depending on CPU and disk speed of
 * your system and performance tradeoffs.
 *
 * Note that each child process consumes between 10 and 50 MB of RAM. If you
 * have limited system resources you may want to use a smaller `maxProcs` value.
 */
exports.exiftool = new ExifTool();
//# sourceMappingURL=ExifTool.js.map