"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Log = exports.logger = exports.setLogger = exports.NoLogger = exports.ConsoleLogger = exports.LogLevels = void 0;
const util_1 = __importDefault(require("util"));
const Object_1 = require("./Object");
const String_1 = require("./String");
exports.LogLevels = [
    "trace",
    "debug",
    "info",
    "warn",
    "error",
];
const _debuglog = util_1.default.debuglog("batch-cluster");
const noop = () => undefined;
/**
 * Default `Logger` implementation.
 *
 * - `debug` and `info` go to {@link util.debuglog}("batch-cluster")`.
 *
 * - `warn` and `error` go to `console.warn` and `console.error`.
 *
 * @see https://nodejs.org/api/util.html#util_util_debuglog_section
 * @see https://nodejs.org/api/console.html
 */
exports.ConsoleLogger = Object.freeze({
    /**
     * No-ops by default, as this is very low-level information.
     */
    trace: noop,
    /**
     * Delegates to `util.debuglog("batch-cluster")`:
     * <https://nodejs.org/api/util.html#util_util_debuglog_section>
     */
    debug: _debuglog,
    /**
     * Delegates to `util.debuglog("batch-cluster")`:
     * <https://nodejs.org/api/util.html#util_util_debuglog_section>
     */
    info: _debuglog,
    /**
     * Delegates to `console.warn`
     */
    warn: console.warn,
    /**
     * Delegates to `console.error`
     */
    error: console.error,
});
/**
 * `Logger` that disables all logging.
 */
exports.NoLogger = Object.freeze({
    trace: noop,
    debug: noop,
    info: noop,
    warn: noop,
    error: noop,
});
let _logger = exports.NoLogger;
function setLogger(l) {
    if (exports.LogLevels.some((ea) => typeof l[ea] !== "function")) {
        throw new Error("invalid logger, must implement " + exports.LogLevels);
    }
    _logger = l;
}
exports.setLogger = setLogger;
function logger() {
    return _logger;
}
exports.logger = logger;
exports.Log = {
    withLevels: (delegate) => {
        const timestamped = {};
        exports.LogLevels.forEach((ea) => {
            const prefix = (ea + " ").substring(0, 5) + " | ";
            timestamped[ea] = (message, ...optionalParams) => {
                if ((0, String_1.notBlank)(message)) {
                    delegate[ea](prefix + message, ...optionalParams);
                }
            };
        });
        return timestamped;
    },
    withTimestamps: (delegate) => {
        const timestamped = {};
        exports.LogLevels.forEach((level) => (timestamped[level] = (message, ...optionalParams) => (0, Object_1.map)(message, (ea) => delegate[level](new Date().toISOString() + " | " + ea, ...optionalParams))));
        return timestamped;
    },
    filterLevels: (l, minLogLevel) => {
        const minLogLevelIndex = exports.LogLevels.indexOf(minLogLevel);
        const filtered = {};
        exports.LogLevels.forEach((ea, idx) => (filtered[ea] = idx < minLogLevelIndex ? noop : l[ea].bind(l)));
        return filtered;
    },
};
//# sourceMappingURL=Logger.js.map