"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExifTime = void 0;
const luxon_1 = require("luxon");
const Maybe_1 = require("./Maybe");
const String_1 = require("./String");
/**
 * Encodes an ExifTime (which may not have a timezone offset)
 */
class ExifTime {
    static fromEXIF(text) {
        if ((0, String_1.blank)(text))
            return;
        text = (0, String_1.toS)(text).trim();
        return (0, Maybe_1.first)(["HH:mm:ss.uZZ", "HH:mm:ssZZ", "HH:mm:ss.u", "HH:mm:ss"], (fmt) => (0, Maybe_1.map)(luxon_1.DateTime.fromFormat(text, fmt), (dt) => this.fromDateTime(dt, text)));
    }
    static fromDateTime(dt, rawValue) {
        return dt == null || !dt.isValid
            ? undefined
            : new ExifTime(dt.hour, dt.minute, dt.second, dt.millisecond, rawValue);
    }
    constructor(hour, minute, second, millisecond, rawValue) {
        this.hour = hour;
        this.minute = minute;
        this.second = second;
        this.millisecond = millisecond;
        this.rawValue = rawValue;
    }
    get millis() {
        return this.millisecond;
    }
    subsec() {
        return this.millisecond == null || this.millisecond === 0
            ? ""
            : "." + (0, String_1.pad3)(this.millisecond);
    }
    toString() {
        return (0, String_1.pad2)(this.hour, this.minute, this.second).join(":") + this.subsec();
    }
    toISOString() {
        return this.toString();
    }
    toExifString() {
        return this.toString();
    }
    toJSON() {
        return {
            _ctor: "ExifTime",
            hour: this.hour,
            minute: this.minute,
            second: this.second,
            millisecond: this.millisecond,
            rawValue: this.rawValue,
        };
    }
    static fromJSON(json) {
        return new ExifTime(json.hour, json.minute, json.second, json.millisecond, json.rawValue);
    }
}
exports.ExifTime = ExifTime;
//# sourceMappingURL=ExifTime.js.map