/// <reference types="node" />
import child_process from "child_process";
import { BatchClusterEmitter, BatchClusterEvents, ChildEndReason, TypedEventEmitter } from "./BatchClusterEmitter";
import { AllOpts, BatchClusterOptions } from "./BatchClusterOptions";
import { WhyNotHealthy, WhyNotReady } from "./BatchProcess";
import { BatchProcessOptions } from "./BatchProcessOptions";
import { Deferred } from "./Deferred";
import { Parser } from "./Parser";
import { Task } from "./Task";
export { BatchClusterOptions } from "./BatchClusterOptions";
export { BatchProcess } from "./BatchProcess";
export { Deferred } from "./Deferred";
export * from "./Logger";
export { SimpleParser } from "./Parser";
export { kill, pidExists, pids } from "./Pids";
export { Rate } from "./Rate";
export { Task } from "./Task";
export type { TypedEventEmitter, BatchClusterEmitter, BatchClusterEvents, BatchProcessOptions, ChildEndReason as ChildExitReason, Parser, WhyNotHealthy, WhyNotReady, };
/**
 * These are required parameters for a given BatchCluster.
 */
export interface ChildProcessFactory {
    /**
     * Expected to be a simple call to execFile. Platform-specific code is the
     * responsibility of this thunk. Error handlers will be registered as
     * appropriate.
     *
     * If this function throws an error or rejects the promise _after_ you've
     * spawned a child process, **the child process may continue to run** and leak
     * system resources.
     */
    readonly processFactory: () => child_process.ChildProcess | Promise<child_process.ChildProcess>;
}
/**
 * BatchCluster instances manage 0 or more homogeneous child processes, and
 * provide the main interface for enqueuing `Task`s via `enqueueTask`.
 *
 * Given the large number of configuration options, the constructor
 * receives a single options hash. The most important of these are the
 * `ChildProcessFactory`, which specifies the factory that creates
 * ChildProcess instances, and `BatchProcessOptions`, which specifies how
 * child tasks can be verified and shut down.
 */
export declare class BatchCluster {
    #private;
    readonly options: AllOpts;
    readonly emitter: BatchClusterEmitter;
    constructor(opts: Partial<BatchClusterOptions> & BatchProcessOptions & ChildProcessFactory);
    /**
     * @see BatchClusterEvents
     */
    readonly on: <E extends keyof BatchClusterEvents>(eventName: E, listener: (...args: BatchClusterEvents[E] extends infer T ? T extends BatchClusterEvents[E] ? T extends (...args: infer A) => void ? A : never : never : never) => void) => BatchClusterEmitter;
    /**
     * @see BatchClusterEvents
     * @since v9.0.0
     */
    readonly off: <E extends keyof BatchClusterEvents>(eventName: E, listener: (...args: BatchClusterEvents[E] extends infer T ? T extends BatchClusterEvents[E] ? T extends (...args: infer A) => void ? A : never : never : never) => void) => BatchClusterEmitter;
    get ended(): boolean;
    /**
     * Shut down this instance, and all child processes.
     * @param gracefully should an attempt be made to finish in-flight tasks, or
     * should we force-kill child PIDs.
     */
    end(gracefully?: boolean): Deferred<void>;
    /**
     * Submits `task` for processing by a `BatchProcess` instance
     *
     * @return a Promise that is resolved or rejected once the task has been
     * attempted on an idle BatchProcess
     */
    enqueueTask<T>(task: Task<T>): Promise<T>;
    /**
     * @return true if all previously-enqueued tasks have settled
     */
    get isIdle(): boolean;
    /**
     * @return the number of pending tasks
     */
    get pendingTaskCount(): number;
    /**
     * @returns {number} the mean number of tasks completed by child processes
     */
    get meanTasksPerProc(): number;
    /**
     * @return the total number of child processes created by this instance
     */
    get spawnedProcCount(): number;
    /**
     * @return the current number of spawned child processes. Some (or all) may be idle.
     */
    get procCount(): number;
    /**
     * @return the current number of child processes currently servicing tasks
     */
    get busyProcCount(): number;
    get startingProcCount(): number;
    /**
     * @return the current pending Tasks (mostly for testing)
     */
    get pendingTasks(): Task<any>[];
    /**
     * @return the current running Tasks (mostly for testing)
     */
    get currentTasks(): Task[];
    /**
     * For integration tests:
     */
    get internalErrorCount(): number;
    /**
     * Verify that each BatchProcess PID is actually alive.
     *
     * @return the spawned PIDs that are still in the process table.
     */
    pids(): Promise<number[]>;
    /**
     * For diagnostics. Contents may change.
     */
    stats(): {
        pendingTaskCount: number;
        currentProcCount: number;
        readyProcCount: number;
        maxProcCount: number;
        internalErrorCount: number;
        startErrorRatePerMinute: number;
        msBeforeNextSpawn: number;
        spawnedProcCount: number;
        childEndCounts: {
            startError: number;
            timeout: number;
            broken: number;
            closed: number;
            ending: number;
            ended: number;
            idle: number;
            old: number;
            "proc.close": number;
            "proc.disconnect": number;
            "proc.error": number;
            "proc.exit": number;
            "stderr.error": number;
            stderr: number;
            "stdin.error": number;
            "stdout.error": number;
            tooMany: number;
            unhealthy: number;
            worn: number;
        };
        ending: boolean;
        ended: boolean;
    };
    /**
     * Get ended process counts (used for tests)
     */
    countEndedChildProcs(why: ChildEndReason): number;
    get childEndCounts(): {
        [key in NonNullable<ChildEndReason>]: number;
    };
    /**
     * Shut down any currently-running child processes. New child processes will
     * be started automatically to handle new tasks.
     */
    closeChildProcesses(gracefully?: boolean): Promise<void>;
    /**
     * Reset the maximum number of active child processes to `maxProcs`. Note that
     * this is handled gracefully: child processes are only reduced as tasks are
     * completed.
     */
    setMaxProcs(maxProcs: number): void;
    /**
     * Run maintenance on currently spawned child processes. This method is
     * normally invoked automatically as tasks are enqueued and processed.
     *
     * Only public for tests.
     */
    vacuumProcs(): Promise<void[]>;
}
